<?php

require_once 'config.php';
require_once 'domain_checker.php';
require_once 'telegram_bot.php';

// Hata raporlamayı aç
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/cron_error.log');

try {
    $pdo = getDBConnection();
    
    // Veritabanındaki tüm domain'leri al
    $stmt = $pdo->query("SELECT * FROM domains ORDER BY id ASC");
    $domains = $stmt->fetchAll();
    
    if (empty($domains)) {
        echo "Veritabanında kontrol edilecek domain bulunamadı.\n";
        exit(0);
    }
    
    foreach ($domains as $domain) {
        $domainId = $domain['id'];
        $domainUrl = $domain['domain_url'];
        echo "Kontrol ediliyor: $domainUrl\n";
        
        // Domain'e giriş yap ve yönlendirmeyi kontrol et
        $checkResult = checkDomainRedirect($domainUrl);
        
        // Veritabanındaki mevcut yönlendirme bilgisini al
        $currentRedirectDomain = $domain['redirect_domain'];
        
        if (!$checkResult['success']) {
            echo "Hata: " . $checkResult['error'] . "\n";
            
            // 403 hatası için özel mesaj
            if (strpos($checkResult['error'], '403') !== false) {
                echo "Uyarı: 403 hatası alındı. Response header ve body kontrol ediliyor...\n";
                echo "Debug bilgileri cron_error.log dosyasına kaydedildi.\n";
            }
            
            // Hata durumunu geçmişe kaydet
            saveDomainHistory($pdo, $domainId, $domainUrl, $currentRedirectDomain, '', 'error', $checkResult['error']);
            continue;
        }
        
        $newRedirectDomain = $checkResult['redirect_domain'];
        echo "Yönlendirilen domain: $newRedirectDomain\n";
        
        // Uyarı varsa göster
        if (isset($checkResult['warning'])) {
            echo "Uyarı: " . $checkResult['warning'] . "\n";
        }
        
        // İlk kontrol ise (redirect_domain null ise) sadece kaydet
        if (empty($currentRedirectDomain)) {
            echo "İlk kontrol - Domain kaydediliyor...\n";
            insertDomainRedirect($pdo, $domainUrl, $newRedirectDomain);
            // İlk kontrolü geçmişe kaydet
            saveDomainHistory($pdo, $domainId, $domainUrl, null, $newRedirectDomain, 'success');
            echo "Domain kaydedildi.\n";
            continue;
        }
        
        // Domain değişmiş mi kontrol et
        if ($currentRedirectDomain !== $newRedirectDomain) {
            echo "Domain değişti! Eski: $currentRedirectDomain -> Yeni: $newRedirectDomain\n";
            
            // Değişikliği geçmişe kaydet
            saveDomainHistory($pdo, $domainId, $domainUrl, $currentRedirectDomain, $newRedirectDomain, 'success');
            
            // Telegram mesajı gönder
            $messageData = parseMessageTemplate($newRedirectDomain);
            
            // Aktif kanal ID'sini al
            $activeChannelId = getActiveChannelId($pdo);
            
            echo "Telegram mesajı gönderiliyor...\n";
            $telegramResult = sendTelegramMessage(
                $messageData['message'],
                $messageData['keyboard'],
                $pdo
            );
            
            if ($telegramResult['success']) {
                echo "Telegram mesajı başarıyla gönderildi.\n";
            } else {
                echo "Telegram mesajı gönderilemedi: " . json_encode($telegramResult) . "\n";
                // Başarısız mesajı kaydet (retry için)
                if ($activeChannelId) {
                    saveFailedMessage(
                        $pdo,
                        $domainId,
                        $domainUrl,
                        $newRedirectDomain,
                        $activeChannelId,
                        $messageData['message'],
                        $messageData['keyboard'],
                        $telegramResult['error'] ?? 'Bilinmeyen hata'
                    );
                }
            }
            
            // Veritabanını güncelle
            updateDomainRedirect($pdo, $domainUrl, $newRedirectDomain);
            echo "Veritabanı güncellendi.\n";
        } else {
            echo "Domain değişmedi, işlem yapılmıyor.\n";
            // Yine de last_check'i güncelle ve geçmişe kaydet (değişiklik yok ama kontrol yapıldı)
            updateDomainRedirect($pdo, $domainUrl, $newRedirectDomain);
            saveDomainHistory($pdo, $domainId, $domainUrl, $currentRedirectDomain, $newRedirectDomain, 'success');
        }
        
        echo "---\n";
    }
    
    // Başarısız mesajları tekrar dene (retry)
    echo "\nBaşarısız mesajlar kontrol ediliyor...\n";
    $failedMessages = getFailedMessages($pdo);
    
    if (!empty($failedMessages)) {
        echo count($failedMessages) . " adet başarısız mesaj bulundu.\n";
        
        foreach ($failedMessages as $failedMsg) {
            echo "Tekrar deneniyor: Domain ID " . $failedMsg['domain_id'] . "\n";
            
            // Durumu retrying olarak güncelle
            updateFailedMessageStatus($pdo, $failedMsg['id'], 'retrying');
            
            // Keyboard'u decode et
            $keyboard = $failedMsg['keyboard_data'] ? json_decode($failedMsg['keyboard_data'], true) : null;
            
            // Mesajı tekrar gönder
            $result = sendTelegramMessageToChannel(
                $failedMsg['channel_id'],
                $failedMsg['message_text'],
                $keyboard,
                $pdo
            );
            
            if ($result['success']) {
                echo "Mesaj başarıyla gönderildi (ID: " . $failedMsg['id'] . ")\n";
                updateFailedMessageStatus($pdo, $failedMsg['id'], 'sent');
            } else {
                $retryCount = $failedMsg['retry_count'] + 1;
                echo "Mesaj gönderilemedi (ID: " . $failedMsg['id'] . ", Deneme: $retryCount/" . $failedMsg['max_retries'] . ")\n";
                
                if ($retryCount >= $failedMsg['max_retries']) {
                    echo "Maksimum deneme sayısına ulaşıldı, mesaj başarısız olarak işaretlendi.\n";
                    updateFailedMessageStatus($pdo, $failedMsg['id'], 'failed', $result['error'] ?? 'Maksimum deneme sayısına ulaşıldı');
                } else {
                    updateFailedMessageStatus($pdo, $failedMsg['id'], 'pending', $result['error'] ?? 'Tekrar denenecek');
                }
            }
        }
    } else {
        echo "Başarısız mesaj bulunamadı.\n";
    }
    
    echo "\nTüm kontroller tamamlandı.\n";
    
} catch (Exception $e) {
    echo "Hata: " . $e->getMessage() . "\n";
    error_log("Cron hatası: " . $e->getMessage());
    exit(1);
}

