import json
import logging
import os
import re
import threading
import time
from datetime import datetime

import requests
from flask import Flask, redirect, render_template_string, request, url_for
from selenium import webdriver
from selenium.common.exceptions import WebDriverException, TimeoutException
from selenium.webdriver.chrome.options import Options as ChromeOptions
from selenium.webdriver.chrome.service import Service as ChromeService

# === YAPILANDIRMA ===
DATA_FILE = "domain_state.json"
CHECK_INTERVAL_SECONDS = 300  # 5 dakika
MESSAGE_TEMPLATE_PATH = "mesaj.md"
TELEGRAM_BOT_TOKEN_FALLBACK = "8548703810:AAEY94q--mIzOdbDRZSv-iRrYkppiAkqr3M"
TELEGRAM_CHAT_ID_FALLBACK = "-1003247510379"
CHROMEDRIVER_PATH = "chromedriver.exe"

logging.basicConfig(
    level=logging.INFO,
    format="%(asctime)s [%(levelname)s] %(message)s",
    datefmt="%Y-%m-%d %H:%M:%S"
)

requests.packages.urllib3.disable_warnings()

USER_AGENT_HEADERS = {
    "User-Agent": (
        "Mozilla/5.0 (Windows NT 10.0; Win64; x64) "
        "AppleWebKit/537.36 (KHTML, like Gecko) "
        "Chrome/120.0.0.0 Safari/537.36"
    ),
    "Accept": (
        "text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,"
        "image/webp,image/apng,*/*;q=0.8"
    ),
    "Accept-Language": "tr-TR,tr;q=0.9,en-US;q=0.8,en;q=0.7",
    "Accept-Encoding": "gzip, deflate",
    "Connection": "keep-alive",
    "Upgrade-Insecure-Requests": "1",
}

state_lock = threading.Lock()
state = {}


def default_state():
    return {
        "next_domain_id": 1,
        "domains": [],
        "telegram": {
            "bot_token": "",
            "chat_id": ""
        }
    }


def ensure_state_defaults(data):
    if not isinstance(data, dict):
        data = default_state()
    data.setdefault("next_domain_id", 1)
    data.setdefault("domains", [])
    data.setdefault("telegram", {})
    telegram = data["telegram"]
    telegram.setdefault("bot_token", "")
    telegram.setdefault("chat_id", "")
    for domain in data["domains"]:
        domain.setdefault("id", data["next_domain_id"])
        domain.setdefault("url", "")
        domain.setdefault("last_redirect", "")
        domain.setdefault("last_checked", "")
        domain.setdefault("history", [])
    return data


def load_state():
    if not os.path.exists(DATA_FILE):
        data = default_state()
        save_state(data)
        return data
    try:
        with open(DATA_FILE, "r", encoding="utf-8") as f:
            data = json.load(f)
    except (json.JSONDecodeError, OSError):
        logging.warning("State dosyası okunamadı, varsayılan ayarlar kullanılacak.")
        data = default_state()
        save_state(data)
        return data
    return ensure_state_defaults(data)


def save_state(new_state=None):
    with state_lock:
        if new_state is not None:
            global state
            state = ensure_state_defaults(new_state)
        with open(DATA_FILE, "w", encoding="utf-8") as f:
            json.dump(state, f, ensure_ascii=False, indent=2)


def get_state_copy():
    with state_lock:
        return json.loads(json.dumps(state))


def find_domain_by_id(domain_id):
    for domain in state["domains"]:
        if domain["id"] == domain_id:
            return domain
    return None


def resolve_url(url):
    session = requests.Session()
    session.headers.update(USER_AGENT_HEADERS)
    try:
        response = session.get(url, allow_redirects=True, timeout=30, verify=False)
        final_url = response.url
        status = response.status_code
        if status >= 400:
            logging.warning("HTTP %s hatası alındı, Selenium ile tekrar denenecek.", status)
            success, selenium_url, selenium_error = resolve_url_with_selenium(url)
            if success:
                return True, selenium_url, None
            msg = selenium_error or (f"HTTP {status} hatası")
            return False, None, msg
        if not final_url:
            logging.warning("Yönlendirme tespit edilemedi, Selenium ile tekrar denenecek.")
            success, selenium_url, selenium_error = resolve_url_with_selenium(url)
            if success:
                return True, selenium_url, None
            msg = selenium_error or "Yönlendirme tespit edilemedi."
            return False, None, msg
        return True, final_url, None
    except requests.RequestException as exc:
        logging.warning("HTTP isteği başarısız oldu, Selenium denenecek: %s", exc)

    # HTTP isteği başarısızsa Selenium ile dene
    return resolve_url_with_selenium(url)


def resolve_url_with_selenium(url):
    driver = None
    options = ChromeOptions()
    options.add_argument("--headless=new")
    options.add_argument("--disable-gpu")
    options.add_argument("--disable-dev-shm-usage")
    options.add_argument("--no-sandbox")
    options.add_argument(f"--user-agent={USER_AGENT_HEADERS['User-Agent']}")
    options.add_argument("--log-level=3")

    service = ChromeService(executable_path=CHROMEDRIVER_PATH)

    try:
        driver = webdriver.Chrome(service=service, options=options)
        driver.set_page_load_timeout(45)
        driver.get(url)
        time.sleep(5)  # yönlendirmelerin tamamlanması için kısa bekleme
        final_url = driver.current_url
        if not final_url or final_url == "data:,":
            return False, None, "Selenium ile yönlendirme tespit edilemedi."
        return True, final_url, None
    except (WebDriverException, TimeoutException) as exc:
        logging.error("Selenium ile yönlendirme alınamadı: %s", exc)
        return False, None, f"Selenium hatası: {exc}"
    finally:
        try:
            if driver:
                driver.quit()
        except Exception:
            pass


def parse_message_template(new_domain):
    try:
        with open(MESSAGE_TEMPLATE_PATH, "r", encoding="utf-8") as f:
            content = f.read()
    except FileNotFoundError:
        return f"Domain güncellendi: {new_domain}", None

    parts = content.split("-----------------")
    message_part = parts[0].replace("https://casinolevant792.com/tr", new_domain).strip()

    keyboard = None
    if len(parts) > 1:
        button_lines = [
            line.strip() for line in parts[1].splitlines() if line.strip()
        ]
        keyboard = build_keyboard(button_lines, new_domain)
    return message_part, keyboard


def build_keyboard(button_lines, new_domain):
    keyboard = []
    for line in button_lines:
        matches = re.findall(r"([^\(]+)\((https?://[^)]+)\)", line)
        row = []
        for label, url in matches:
            label_clean = re.sub(r"\s+", " ", label).strip()
            url_final = url.replace("https://casinolevant792.com/tr", new_domain)
            row.append({"text": label_clean, "url": url_final})
        if row:
            keyboard.append(row)
    return keyboard if keyboard else None


def get_telegram_credentials():
    with state_lock:
        bot_token = state["telegram"].get("bot_token") or TELEGRAM_BOT_TOKEN_FALLBACK
        chat_id = state["telegram"].get("chat_id") or TELEGRAM_CHAT_ID_FALLBACK
    if not bot_token or bot_token == "YOUR_BOT_TOKEN_HERE":
        return None, None
    if not chat_id or chat_id == "YOUR_CHAT_ID_HERE":
        return None, None
    return bot_token, chat_id


def send_telegram_message(message, keyboard=None):
    bot_token, chat_id = get_telegram_credentials()
    if not bot_token or not chat_id:
        logging.warning("Telegram ayarları eksik, mesaj gönderilmeyecek.")
        return False

    payload = {
        "chat_id": chat_id,
        "text": message,
        "parse_mode": "HTML",
        "disable_web_page_preview": False
    }
    if keyboard:
        payload["reply_markup"] = {"inline_keyboard": keyboard}

    try:
        response = requests.post(
            f"https://api.telegram.org/bot{bot_token}/sendMessage",
            json=payload,
            timeout=20
        )
        if response.ok:
            logging.info("Telegram mesajı gönderildi.")
            return True
        else:
            logging.error("Telegram mesajı gönderilemedi: HTTP %s - %s",
                          response.status_code, response.text)
            return False
    except requests.RequestException as exc:
        logging.error("Telegram mesajı gönderilemedi: %s", exc)
        return False


def add_history_entry(domain, status, old_redirect, new_redirect, error=None):
    entry = {
        "timestamp": datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
        "status": status,
        "old_redirect": old_redirect,
        "new_redirect": new_redirect,
        "error": error
    }
    history = domain.setdefault("history", [])
    history.append(entry)
    if len(history) > 50:
        del history[:-50]


def run_check_cycle():
    with state_lock:
        domain_snapshots = [domain.copy() for domain in state["domains"]]

    notifications = []

    for snapshot in domain_snapshots:
        domain_id = snapshot["id"]
        url = snapshot["url"]
        logging.info("Kontrol ediliyor: %s", url)
        success, new_redirect, error = resolve_url(url)

        with state_lock:
            domain_obj = find_domain_by_id(domain_id)
            if not domain_obj:
                continue
            old_redirect = domain_obj.get("last_redirect", "")
            if success:
                domain_obj["last_redirect"] = new_redirect
                domain_obj["last_checked"] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                changed = (new_redirect != old_redirect)
                add_history_entry(domain_obj, "success", old_redirect, new_redirect)
                write_state_locked()
            else:
                add_history_entry(domain_obj, "error", old_redirect, old_redirect, error)
                domain_obj["last_checked"] = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
                write_state_locked()
                changed = False

        if success and changed:
            message, keyboard = parse_message_template(new_redirect)
            notifications.append((message, keyboard))

    for message, keyboard in notifications:
        send_telegram_message(message, keyboard)


def write_state_locked():
    with open(DATA_FILE, "w", encoding="utf-8") as f:
        json.dump(state, f, ensure_ascii=False, indent=2)


def background_worker():
    logging.info("Domain izleme iş parçacığı başlatıldı.")
    while True:
        try:
            run_check_cycle()
        except Exception:
            logging.exception("Domain kontrol döngüsü sırasında hata oluştu.")
        time.sleep(CHECK_INTERVAL_SECONDS)


app = Flask(__name__)


INDEX_TEMPLATE = """
<!doctype html>
<html lang="tr">
<head>
  <meta charset="utf-8">
  <title>Domain İzleme Paneli</title>
  <style>
    body { font-family: Arial, sans-serif; background: #f5f5f5; margin: 0; padding: 0; }
    .container { max-width: 1100px; margin: 0 auto; padding: 30px; }
    h1 { margin-bottom: 20px; }
    table { width: 100%; border-collapse: collapse; background: #fff; }
    th, td { padding: 10px 12px; border: 1px solid #ddd; text-align: left; }
    th { background: #f0f0f0; }
    .section { background: #fff; padding: 20px; border-radius: 8px; margin-bottom: 20px; box-shadow: 0 2px 4px rgba(0,0,0,0.05); }
    .form-row { display: flex; gap: 10px; align-items: center; margin-bottom: 10px; flex-wrap: wrap; }
    input[type="text"] { padding: 8px; border: 1px solid #ccc; border-radius: 4px; flex: 1; }
    button { padding: 8px 16px; border: none; border-radius: 4px; cursor: pointer; }
    .btn-primary { background: #007bff; color: #fff; }
    .btn-danger { background: #dc3545; color: #fff; }
    .btn-secondary { background: #6c757d; color: #fff; }
    .alert { padding: 10px 15px; background: #e9ecef; border-radius: 4px; margin-bottom: 15px; }
    .history { font-size: 0.9em; color: #555; }
    .history-item { border-top: 1px dashed #ddd; padding-top: 6px; margin-top: 6px; }
    .small-btn { padding: 4px 8px; font-size: 0.85em; }
  </style>
</head>
<body>
  <div class="container">
    <h1>Domain İzleme Paneli</h1>
    {% if message %}
      <div class="alert">{{ message }}</div>
    {% endif %}

    <div class="section">
      <h2>Telegram Ayarları</h2>
      <form method="post" action="{{ url_for('update_telegram') }}">
        <div class="form-row">
          <label>Bot Token</label>
          <input type="text" name="bot_token" value="{{ telegram.bot_token }}">
        </div>
        <div class="form-row">
          <label>Chat ID</label>
          <input type="text" name="chat_id" value="{{ telegram.chat_id }}">
        </div>
        <button class="btn-primary" type="submit">Kaydet</button>
      </form>
    </div>

    <div class="section">
      <h2>Domainler</h2>
      <form method="post" action="{{ url_for('add_domain') }}">
        <div class="form-row">
          <input type="text" name="domain_url" placeholder="https://ornek.com" required>
          <button class="btn-primary" type="submit">Domain Ekle</button>
        </div>
      </form>
      <table>
        <thead>
          <tr>
            <th>ID</th>
            <th>Domain URL</th>
            <th>Son Yönlendirme</th>
            <th>Son Kontrol</th>
            <th>Geçmiş</th>
            <th>İşlem</th>
          </tr>
        </thead>
        <tbody>
          {% for domain in domains %}
          <tr>
            <td>{{ domain.id }}</td>
            <td><code>{{ domain.url }}</code></td>
            <td>
              {% if domain.last_redirect %}
                <code>{{ domain.last_redirect }}</code>
              {% else %}
                <em>-</em>
              {% endif %}
            </td>
            <td>{{ domain.last_checked or '-' }}</td>
            <td class="history">
              {% if domain.history_preview %}
                {% for item in domain.history_preview %}
                  <div class="history-item">
                    <strong>{{ item.timestamp }}</strong><br>
                    Durum: {{ item.status }}
                    {% if item.error %}<br>Hata: {{ item.error }}{% endif %}
                    <br>Eski: <code>{{ item.old_redirect or '-' }}</code>
                    <br>Yeni: <code>{{ item.new_redirect or '-' }}</code>
                  </div>
                {% endfor %}
              {% else %}
                <em>Geçmiş yok</em>
              {% endif %}
            </td>
            <td>
              <form method="post" action="{{ url_for('delete_domain', domain_id=domain.id) }}" onsubmit="return confirm('Domain silinsin mi?');">
                <button class="btn-danger small-btn" type="submit">Sil</button>
              </form>
            </td>
          </tr>
          {% endfor %}
        </tbody>
      </table>
    </div>

    <div class="section">
      <h2>Manuel Kontrol</h2>
      <form method="post" action="{{ url_for('run_check_now') }}">
        <button class="btn-secondary" type="submit">Şimdi Kontrol Et</button>
      </form>
    </div>
  </div>
</body>
</html>
"""


@app.route("/")
def index():
    message = request.args.get("msg", "")
    data = get_state_copy()
    domains = data["domains"]
    # Her domain için son 5 geçmiş kaydını göster (yeniden eskiye)
    for domain in domains:
        history = domain.get("history", [])
        domain["history_preview"] = list(reversed(history[-5:]))
    return render_template_string(
        INDEX_TEMPLATE,
        domains=domains,
        telegram=data["telegram"],
        message=message
    )


@app.route("/add-domain", methods=["POST"])
def add_domain():
    url = request.form.get("domain_url", "").strip()
    if not url or not url.lower().startswith(("http://", "https://")):
        return redirect(url_for("index", msg="Geçerli bir URL girin."))

    with state_lock:
        global state
        domain_id = state["next_domain_id"]
        state["next_domain_id"] += 1
        state["domains"].append({
            "id": domain_id,
            "url": url,
            "last_redirect": "",
            "last_checked": "",
            "history": []
        })
        write_state_locked()

    logging.info("Yeni domain eklendi: %s", url)
    return redirect(url_for("index", msg="Domain eklendi."))


@app.route("/delete-domain/<int:domain_id>", methods=["POST"])
def delete_domain(domain_id):
    with state_lock:
        global state
        state["domains"] = [d for d in state["domains"] if d["id"] != domain_id]
        write_state_locked()
    logging.info("Domain silindi: %s", domain_id)
    return redirect(url_for("index", msg="Domain silindi."))


@app.route("/update-telegram", methods=["POST"])
def update_telegram():
    bot_token = request.form.get("bot_token", "").strip()
    chat_id = request.form.get("chat_id", "").strip()
    with state_lock:
        state["telegram"]["bot_token"] = bot_token
        state["telegram"]["chat_id"] = chat_id
        write_state_locked()
    logging.info("Telegram ayarları güncellendi.")
    return redirect(url_for("index", msg="Telegram ayarları güncellendi."))


@app.route("/run-check", methods=["POST"])
def run_check_now():
    run_check_cycle()
    return redirect(url_for("index", msg="Kontrol tamamlandı."))


def start_background_thread():
    thread = threading.Thread(target=background_worker, daemon=True)
    thread.start()


def main():
    global state
    state = load_state()
    start_background_thread()
    logging.info("Admin paneli http://127.0.0.1:5000 üzerinde çalışıyor.")
    app.run(host="127.0.0.1", port=5000, debug=False, threaded=True)


if __name__ == "__main__":
    main()

