<?php
require_once 'config.php';

/**
 * Domain'e giriş yapıp yönlendirilen domain'i kontrol eder
 * @param string $domainUrl Kontrol edilecek domain URL'i
 * @return array ['redirect_domain' => string, 'success' => bool, 'error' => string]
 */
function checkDomainRedirect($domainUrl) {
    // t.ly kısa linkleri için API üzerinden genişletme denemesi
    $parsedInputUrl = parse_url($domainUrl);
    if ($parsedInputUrl && isset($parsedInputUrl['host'])) {
        $host = strtolower($parsedInputUrl['host']);
        if (in_array($host, ['t.ly', 'www.t.ly', 'tly.link', 'www.tly.link'])) {
            $expandResult = expandTlyShortUrl($domainUrl);
            if ($expandResult['success']) {
                return [
                    'success' => true,
                    'redirect_domain' => $expandResult['redirect_domain'],
                    'error' => null
                ];
            }
            return [
                'success' => false,
                'redirect_domain' => null,
                'error' => $expandResult['error'] ?? 't.ly API genişletme hatası'
            ];
        }
    }

    $ch = curl_init();
    
    // Gerçekçi tarayıcı header'ları (array olarak saklanacak - ikinci istek için)
    $headersArray = [
        'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
        'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,image/apng,*/*;q=0.8',
        'Accept-Language: tr-TR,tr;q=0.9,en-US;q=0.8,en;q=0.7',
        'Accept-Encoding: gzip, deflate', // Brotli (br) curl tarafından desteklenmiyor, kaldırıldı
        'Connection: keep-alive',
        'Upgrade-Insecure-Requests: 1',
        'Sec-Fetch-Dest: document',
        'Sec-Fetch-Mode: navigate',
        'Sec-Fetch-Site: none',
        'Cache-Control: max-age=0'
    ];
    
    curl_setopt_array($ch, [
        CURLOPT_URL => $domainUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => false, // 403 hatası alındığında FOLLOWLOCATION çalışmayabilir, manuel kontrol edeceğiz
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_SSL_VERIFYHOST => false,
        CURLOPT_HTTPHEADER => $headersArray,
        CURLOPT_ENCODING => 'gzip, deflate', // Sadece curl'un desteklediği encoding'ler
        CURLOPT_HEADER => true,
        CURLOPT_NOBODY => false,
        CURLOPT_COOKIEFILE => '', // Cookie desteği
        CURLOPT_COOKIEJAR => '',
        CURLOPT_AUTOREFERER => true, // Otomatik referer
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $effectiveUrl = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
    $redirectUrl = curl_getinfo($ch, CURLINFO_REDIRECT_URL);
    $error = curl_error($ch);
    
    // Header'ları parse et (Location header'ını bulmak için)
    $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
    $headers = substr($response, 0, $headerSize);
    $body = substr($response, $headerSize);
    
    curl_close($ch);
    
    // Debug: 403 hatası alındığında header'ları logla
    if ($httpCode == 403) {
        error_log("403 Hatası - Domain: $domainUrl");
        error_log("Headers: " . substr($headers, 0, 500));
        error_log("Body (ilk 500 karakter): " . substr($body, 0, 500));
    }
    
    // Location header'ını bul (case-insensitive, tüm satırları kontrol et)
    $locationUrl = null;
    
    // Önce standart Location header'ını kontrol et
    if (preg_match('/^Location:\s*(.+?)$/mi', $headers, $matches)) {
        $locationUrl = trim($matches[1]);
        // Relative URL ise absolute URL'e çevir
        if (!preg_match('/^https?:\/\//', $locationUrl)) {
            $baseUrl = parse_url($domainUrl);
            $locationUrl = $baseUrl['scheme'] . '://' . $baseUrl['host'] . 
                          (isset($baseUrl['port']) ? ':' . $baseUrl['port'] : '') .
                          (strpos($locationUrl, '/') === 0 ? $locationUrl : '/' . $locationUrl);
        }
    }
    
    // Eğer Location header yoksa, tüm header satırlarında "location" kelimesini ara
    if (empty($locationUrl)) {
        $headerLines = explode("\n", $headers);
        foreach ($headerLines as $line) {
            if (stripos($line, 'location') !== false && stripos($line, ':') !== false) {
                $parts = explode(':', $line, 2);
                if (count($parts) == 2 && stripos(trim($parts[0]), 'location') !== false) {
                    $locationUrl = trim($parts[1]);
                    if (!preg_match('/^https?:\/\//', $locationUrl)) {
                        $baseUrl = parse_url($domainUrl);
                        $locationUrl = $baseUrl['scheme'] . '://' . $baseUrl['host'] . 
                                      (isset($baseUrl['port']) ? ':' . $baseUrl['port'] : '') .
                                      (strpos($locationUrl, '/') === 0 ? $locationUrl : '/' . $locationUrl);
                    }
                    break;
                }
            }
        }
    }
    
    // Eğer Location header yoksa, response body'de meta refresh veya JavaScript redirect kontrol et
    if (empty($locationUrl) && !empty($body)) {
        // Meta refresh kontrolü: <meta http-equiv="refresh" content="0;url=...">
        if (preg_match('/<meta[^>]+http-equiv=["\']?refresh["\']?[^>]+content=["\']?[^;]*;\s*url=([^"\'>\s]+)/i', $body, $matches)) {
            $locationUrl = trim($matches[1]);
            if (!preg_match('/^https?:\/\//', $locationUrl)) {
                $baseUrl = parse_url($domainUrl);
                $locationUrl = $baseUrl['scheme'] . '://' . $baseUrl['host'] . 
                              (isset($baseUrl['port']) ? ':' . $baseUrl['port'] : '') .
                              (strpos($locationUrl, '/') === 0 ? $locationUrl : '/' . $locationUrl);
            }
        }
        // JavaScript redirect kontrolü: window.location = "..." veya window.location.href = "..."
        elseif (preg_match('/window\.location(?:\.href)?\s*=\s*["\']([^"\']+)["\']/i', $body, $matches)) {
            $locationUrl = trim($matches[1]);
            if (!preg_match('/^https?:\/\//', $locationUrl)) {
                $baseUrl = parse_url($domainUrl);
                $locationUrl = $baseUrl['scheme'] . '://' . $baseUrl['host'] . 
                              (isset($baseUrl['port']) ? ':' . $baseUrl['port'] : '') .
                              (strpos($locationUrl, '/') === 0 ? $locationUrl : '/' . $locationUrl);
            }
        }
        // JavaScript redirect alternatif: location.replace("...")
        elseif (preg_match('/location\.replace\(["\']([^"\']+)["\']\)/i', $body, $matches)) {
            $locationUrl = trim($matches[1]);
            if (!preg_match('/^https?:\/\//', $locationUrl)) {
                $baseUrl = parse_url($domainUrl);
                $locationUrl = $baseUrl['scheme'] . '://' . $baseUrl['host'] . 
                              (isset($baseUrl['port']) ? ':' . $baseUrl['port'] : '') .
                              (strpos($locationUrl, '/') === 0 ? $locationUrl : '/' . $locationUrl);
            }
        }
        // Body'de direkt URL ara (https:// ile başlayan URL'ler) - Cloudflare domain'lerini filtrele
        elseif (preg_match_all('/https?:\/\/[^\s"\'<>\)]+/i', $body, $allMatches)) {
            // Tüm URL'leri kontrol et, Cloudflare domain'lerini filtrele
            foreach ($allMatches[0] as $foundUrl) {
                $foundUrl = trim($foundUrl);
                $parsedFound = parse_url($foundUrl);
                
                if ($parsedFound && isset($parsedFound['host'])) {
                    $host = strtolower($parsedFound['host']);
                    
                    // Cloudflare ve t.ly domain'lerini filtrele
                    $excludedDomains = [
                        'cloudflare',
                        'cloudflareinsights',
                        'cloudflare.com',
                        'cf-',
                        't.ly',
                        'tly.link',
                        'challenges.cloudflare.com'
                    ];
                    
                    $isExcluded = false;
                    foreach ($excludedDomains as $excluded) {
                        if (strpos($host, $excluded) !== false) {
                            $isExcluded = true;
                            break;
                        }
                    }
                    
                    // Eğer exclude edilmediyse ve geçerli bir domain ise kullan
                    if (!$isExcluded && !empty($parsedFound['host'])) {
                        $locationUrl = $foundUrl;
                        break; // İlk geçerli URL'i kullan
                    }
                }
            }
        }
    }
    
    // Cloudflare challenge sayfası kontrolü
    $isCloudflareChallenge = false;
    if ($httpCode == 403 && (stripos($body, 'Just a moment') !== false || stripos($body, 'cf-mitigated') !== false || stripos($headers, 'cf-mitigated') !== false)) {
        $isCloudflareChallenge = true;
        error_log("Cloudflare challenge sayfası tespit edildi: $domainUrl");
        
        // Cloudflare challenge sayfasında redirect URL'i aramaya çalış
        // Challenge sayfasında genellikle form action veya JavaScript'te URL olabilir
        if (preg_match('/action=["\']([^"\']+)["\']/i', $body, $matches)) {
            $actionUrl = trim($matches[1]);
            if (preg_match('/^https?:\/\//', $actionUrl)) {
                $parsedAction = parse_url($actionUrl);
                if ($parsedAction && isset($parsedAction['host'])) {
                    $host = strtolower($parsedAction['host']);
                    // Cloudflare domain'i değilse kullan
                    if (strpos($host, 'cloudflare') === false && strpos($host, 't.ly') === false) {
                        $locationUrl = $actionUrl;
                    }
                }
            }
        }
        
        // Body'de direkt URL ara (challenge sayfasında da olabilir) - Cloudflare domain'lerini filtrele
        if (empty($locationUrl) && preg_match_all('/https?:\/\/[^\s"\'<>\)]+/i', $body, $allMatches)) {
            // Tüm URL'leri kontrol et, Cloudflare domain'lerini filtrele
            foreach ($allMatches[0] as $foundUrl) {
                $foundUrl = trim($foundUrl);
                $parsedFound = parse_url($foundUrl);
                
                if ($parsedFound && isset($parsedFound['host'])) {
                    $host = strtolower($parsedFound['host']);
                    
                    // Cloudflare ve t.ly domain'lerini filtrele
                    $excludedDomains = [
                        'cloudflare',
                        'cloudflareinsights',
                        'cloudflare.com',
                        'cf-',
                        't.ly',
                        'tly.link',
                        'challenges.cloudflare.com',
                        'static.cloudflareinsights.com',
                        'cdn-cgi',
                        'ray-id'
                    ];
                    
                    $isExcluded = false;
                    foreach ($excludedDomains as $excluded) {
                        if (strpos($host, $excluded) !== false) {
                            $isExcluded = true;
                            break;
                        }
                    }
                    
                    // Eğer exclude edilmediyse ve geçerli bir domain ise kullan
                    if (!$isExcluded && !empty($parsedFound['host'])) {
                        // Ek kontrol: URL'in uzunluğu ve path'i kontrol et (script URL'leri genelde uzun path'e sahip)
                        $pathLength = strlen($parsedFound['path'] ?? '');
                        // Eğer path çok uzunsa (script URL'i olabilir) veya .js/.css gibi dosya uzantısı varsa atla
                        if ($pathLength < 100 && !preg_match('/\.(js|css|png|jpg|gif|ico|woff|woff2|ttf|svg)$/i', $parsedFound['path'] ?? '')) {
                            $locationUrl = $foundUrl;
                            break; // İlk geçerli URL'i kullan
                        }
                    }
                }
            }
        }
    }
    
    // Location header bulunduysa, o URL'e ikinci bir istek yap (redirect'i takip et)
    // Cloudflare challenge durumunda ikinci istek yapma (challenge sayfasından gelen URL'ler genelde geçersiz)
    if (!empty($locationUrl) && !$isCloudflareChallenge && ($httpCode == 301 || $httpCode == 302 || $httpCode == 303 || $httpCode == 307 || $httpCode == 308)) {
        // İkinci istek - redirect URL'ine git
        $ch2 = curl_init();
        curl_setopt_array($ch2, [
            CURLOPT_URL => $locationUrl,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_MAXREDIRS => 5,
            CURLOPT_TIMEOUT => 20,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
            CURLOPT_HTTPHEADER => $headersArray,
            CURLOPT_ENCODING => 'gzip, deflate',
            CURLOPT_HEADER => false,
            CURLOPT_NOBODY => false,
            CURLOPT_AUTOREFERER => true,
        ]);
        
        curl_exec($ch2);
        $finalUrl = curl_getinfo($ch2, CURLINFO_EFFECTIVE_URL);
        $finalHttpCode = curl_getinfo($ch2, CURLINFO_HTTP_CODE);
        curl_close($ch2);
        
        if (!empty($finalUrl) && $finalUrl !== $locationUrl) {
            $parsedUrl = parse_url($finalUrl);
            if ($parsedUrl && isset($parsedUrl['host'])) {
                $redirectDomain = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
                if (isset($parsedUrl['path']) && $parsedUrl['path'] !== '/') {
                    $redirectDomain .= $parsedUrl['path'];
                }
                
                return [
                    'success' => true,
                    'redirect_domain' => $redirectDomain,
                    'error' => null,
                    'warning' => $httpCode == 403 ? '403 hatası alındı ancak redirect takip edildi' : null
                ];
            }
        }
    }
    
    // Redirect URL'i kontrol et (öncelik sırası: Location header > redirectUrl > effectiveUrl)
    $finalRedirectUrl = null;
    if (!empty($locationUrl)) {
        // Cloudflare challenge durumunda, bulunan URL'in geçerli olup olmadığını kontrol et
        if ($isCloudflareChallenge) {
            $parsedLocation = parse_url($locationUrl);
            if ($parsedLocation && isset($parsedLocation['host'])) {
                $host = strtolower($parsedLocation['host']);
                // Eğer Cloudflare domain'i ise geçersiz say
                if (strpos($host, 'cloudflare') !== false || strpos($host, 'cf-') !== false || 
                    strpos($host, 'cloudflareinsights') !== false || strpos($host, 'cdn-cgi') !== false) {
                    $locationUrl = null; // Geçersiz URL, kullanma
                }
            }
        }
        $finalRedirectUrl = $locationUrl;
    } elseif (!empty($redirectUrl)) {
        $finalRedirectUrl = $redirectUrl;
    } elseif (!empty($effectiveUrl) && $effectiveUrl !== $domainUrl) {
        $finalRedirectUrl = $effectiveUrl;
    }
    
    // Redirect tespit edildiyse domain'i al
    if (!empty($finalRedirectUrl)) {
        $parsedUrl = parse_url($finalRedirectUrl);
        if ($parsedUrl && isset($parsedUrl['host'])) {
            $host = strtolower($parsedUrl['host']);
            
            // Cloudflare domain kontrolü
            if (strpos($host, 'cloudflare') !== false || strpos($host, 'cf-') !== false || 
                strpos($host, 'cloudflareinsights') !== false || strpos($host, 'cdn-cgi') !== false) {
                // Cloudflare domain'i, geçersiz say
                if ($isCloudflareChallenge) {
                    // Cloudflare challenge durumunda geçerli URL bulunamadı
                    return [
                        'success' => false,
                        'error' => "HTTP Code: 403 - Cloudflare challenge sayfası tespit edildi. Bot koruması aktif, gerçek redirect URL'i tespit edilemedi. Manuel kontrol gerekebilir.",
                        'redirect_domain' => null
                    ];
                }
            }
            
            $redirectDomain = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
            if (isset($parsedUrl['path']) && $parsedUrl['path'] !== '/') {
                $redirectDomain .= $parsedUrl['path'];
            }
            
            // 403 hatası olsa bile redirect tespit edildiyse başarılı say (Cloudflare challenge değilse)
            if ($httpCode == 403 && !$isCloudflareChallenge) {
                return [
                    'success' => true,
                    'redirect_domain' => $redirectDomain,
                    'error' => null,
                    'warning' => '403 hatası alındı ancak redirect tespit edildi (Location: ' . $finalRedirectUrl . ')'
                ];
            }
            
            return [
                'success' => true,
                'redirect_domain' => $redirectDomain,
                'error' => null
            ];
        }
    }
    
    if ($error) {
        return [
            'success' => false,
            'error' => $error,
            'redirect_domain' => null
        ];
    }
    
    if ($httpCode >= 200 && $httpCode < 400) {
        // Effective URL'i temizle ve domain'i çıkar
        $parsedUrl = parse_url($effectiveUrl);
        if ($parsedUrl && isset($parsedUrl['host'])) {
            $redirectDomain = $parsedUrl['scheme'] . '://' . $parsedUrl['host'];
            if (isset($parsedUrl['path']) && $parsedUrl['path'] !== '/') {
                $redirectDomain .= $parsedUrl['path'];
            }
            
            return [
                'success' => true,
                'redirect_domain' => $redirectDomain,
                'error' => null
            ];
        }
    }
    
    // Cloudflare challenge sayfası varsa özel mesaj
    if ($isCloudflareChallenge) {
        return [
            'success' => false,
            'error' => "HTTP Code: 403 - Cloudflare challenge sayfası tespit edildi. Bot koruması aktif, manuel kontrol gerekebilir.",
            'redirect_domain' => null
        ];
    }
    
    return [
        'success' => false,
        'error' => "HTTP Code: $httpCode",
        'redirect_domain' => null
    ];
}

/**
 * Veritabanından domain bilgilerini alır
 * @param PDO $pdo Veritabanı bağlantısı
 * @param string $domainUrl Domain URL'i
 * @return array|null Domain bilgileri veya null
 */
function getDomainInfo($pdo, $domainUrl) {
    $stmt = $pdo->prepare("SELECT * FROM domains WHERE domain_url = ?");
    $stmt->execute([$domainUrl]);
    return $stmt->fetch();
}

/**
 * Domain yönlendirme bilgisini günceller
 * @param PDO $pdo Veritabanı bağlantısı
 * @param string $domainUrl Domain URL'i
 * @param string $redirectDomain Yönlendirilen domain
 * @return bool Başarılı mı?
 */
function updateDomainRedirect($pdo, $domainUrl, $redirectDomain) {
    $stmt = $pdo->prepare("
        UPDATE domains 
        SET redirect_domain = ?, last_check = NOW() 
        WHERE domain_url = ?
    ");
    return $stmt->execute([$redirectDomain, $domainUrl]);
}

/**
 * Domain yönlendirme bilgisini ilk kez kaydeder
 * @param PDO $pdo Veritabanı bağlantısı
 * @param string $domainUrl Domain URL'i
 * @param string $redirectDomain Yönlendirilen domain
 * @return bool Başarılı mı?
 */
function insertDomainRedirect($pdo, $domainUrl, $redirectDomain) {
    $stmt = $pdo->prepare("
        INSERT INTO domains (domain_url, redirect_domain, last_check) 
        VALUES (?, ?, NOW())
        ON DUPLICATE KEY UPDATE 
            redirect_domain = VALUES(redirect_domain),
            last_check = VALUES(last_check)
    ");
    return $stmt->execute([$domainUrl, $redirectDomain]);
}

/**
 * Domain kontrol geçmişini kaydeder
 * @param PDO $pdo Veritabanı bağlantısı
 * @param int $domainId Domain ID
 * @param string $domainUrl Domain URL
 * @param string|null $oldRedirectDomain Eski yönlendirilen domain
 * @param string $newRedirectDomain Yeni yönlendirilen domain
 * @param string $status Kontrol durumu (success, error)
 * @param string|null $errorMessage Hata mesajı (varsa)
 * @return bool Başarılı mı?
 */
function saveDomainHistory($pdo, $domainId, $domainUrl, $oldRedirectDomain, $newRedirectDomain, $status = 'success', $errorMessage = null) {
    try {
        $stmt = $pdo->prepare("
            INSERT INTO domain_history 
            (domain_id, domain_url, old_redirect_domain, new_redirect_domain, check_status, error_message)
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        return $stmt->execute([
            $domainId,
            $domainUrl,
            $oldRedirectDomain,
            $newRedirectDomain,
            $status,
            $errorMessage
        ]);
    } catch (PDOException $e) {
        error_log("Domain history kayıt hatası: " . $e->getMessage());
        return false;
    }
}

/**
 * Başarısız mesajı kaydeder
 * @param PDO $pdo Veritabanı bağlantısı
 * @param int $domainId Domain ID
 * @param string $domainUrl Domain URL
 * @param string $redirectDomain Yönlendirilen domain
 * @param string $channelId Telegram channel ID
 * @param string $messageText Mesaj metni
 * @param array|null $keyboard Butonlar
 * @param string $errorMessage Hata mesajı
 * @return bool Başarılı mı?
 */
function saveFailedMessage($pdo, $domainId, $domainUrl, $redirectDomain, $channelId, $messageText, $keyboard = null, $errorMessage = null) {
    try {
        $keyboardJson = $keyboard ? json_encode($keyboard) : null;
        $stmt = $pdo->prepare("
            INSERT INTO failed_messages 
            (domain_id, domain_url, redirect_domain, channel_id, message_text, keyboard_data, error_message, status)
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending')
        ");
        return $stmt->execute([
            $domainId,
            $domainUrl,
            $redirectDomain,
            $channelId,
            $messageText,
            $keyboardJson,
            $errorMessage
        ]);
    } catch (PDOException $e) {
        error_log("Failed message kayıt hatası: " . $e->getMessage());
        return false;
    }
}

/**
 * Başarısız mesajları alır (retry için)
 * @param PDO $pdo Veritabanı bağlantısı
 * @return array Başarısız mesajlar
 */
function getFailedMessages($pdo) {
    try {
        $stmt = $pdo->query("
            SELECT * FROM failed_messages 
            WHERE status IN ('pending', 'retrying') 
            AND retry_count < max_retries
            ORDER BY created_at ASC
            LIMIT 10
        ");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        error_log("Failed messages okuma hatası: " . $e->getMessage());
        return [];
    }
}

/**
 * Başarısız mesajın durumunu günceller
 * @param PDO $pdo Veritabanı bağlantısı
 * @param int $messageId Mesaj ID
 * @param string $status Yeni durum (sent, failed, retrying)
 * @param string|null $errorMessage Hata mesajı (varsa)
 * @return bool Başarılı mı?
 */
function updateFailedMessageStatus($pdo, $messageId, $status, $errorMessage = null) {
    try {
        if ($status === 'retrying') {
            $stmt = $pdo->prepare("
                UPDATE failed_messages 
                SET status = ?, retry_count = retry_count + 1, last_retry_at = NOW(), error_message = ?
                WHERE id = ?
            ");
            $stmt->execute([$status, $errorMessage, $messageId]);
        } else {
            $stmt = $pdo->prepare("
                UPDATE failed_messages 
                SET status = ?, error_message = ?
                WHERE id = ?
            ");
            $stmt->execute([$status, $errorMessage, $messageId]);
        }
        return true;
    } catch (PDOException $e) {
        error_log("Failed message güncelleme hatası: " . $e->getMessage());
        return false;
    }
}

/**
 * t.ly kısa linkini API üzerinden genişletir
 * @param string $shortUrl
 * @return array
 */
function expandTlyShortUrl($shortUrl) {
    try {
        $pdo = getDBConnection();
    } catch (Exception $e) {
        return [
            'success' => false,
            'redirect_domain' => null,
            'error' => 'Veritabanı bağlantı hatası: ' . $e->getMessage()
        ];
    }

    $apiKey = getTlyApiKey($pdo);
    if (empty($apiKey)) {
        return [
            'success' => false,
            'redirect_domain' => null,
            'error' => 't.ly API anahtarı bulunamadı. Lütfen config.php veya settings tablosundan ekleyin.'
        ];
    }

    $parsedShort = parse_url($shortUrl);
    $slug = null;
    if ($parsedShort && isset($parsedShort['path'])) {
        $slug = ltrim($parsedShort['path'], '/');
    }

    $payload = [
        'short_url' => $shortUrl
    ];
    if ($slug) {
        $payload['code'] = $slug;
        $payload['slug'] = $slug;
    }
    $payloadJson = json_encode($payload);

    $ch = curl_init('https://api.t.ly/api/v1/link/expand');
    curl_setopt_array($ch, [
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $payloadJson,
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/json',
            'Accept: application/json',
            'Authorization: Bearer ' . $apiKey
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 20,
        CURLOPT_CONNECTTIMEOUT => 10
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($curlError) {
        return [
            'success' => false,
            'redirect_domain' => null,
            'error' => 't.ly API bağlantı hatası: ' . $curlError
        ];
    }

    if ($httpCode >= 200 && $httpCode < 300) {
        $data = json_decode($response, true);
        if (is_array($data)) {
            $candidates = [
                $data['destination_url'] ?? null,
                $data['destination'] ?? null,
                $data['long_url'] ?? null,
                $data['url'] ?? null
            ];
            foreach ($candidates as $candidate) {
                if (!empty($candidate) && filter_var($candidate, FILTER_VALIDATE_URL)) {
                    return [
                        'success' => true,
                        'redirect_domain' => $candidate,
                        'error' => null
                    ];
                }
            }
        }
        error_log("t.ly API beklenmeyen yanıt: " . $response);
        return [
            'success' => false,
            'redirect_domain' => null,
            'error' => 't.ly API geçerli bir yanıt döndürmedi.'
        ];
    }

    error_log("t.ly API hatası (HTTP $httpCode): $response");
    return [
        'success' => false,
        'redirect_domain' => null,
        'error' => 't.ly API hatası (HTTP ' . $httpCode . ')'
    ];
}

