<?php
require_once 'config.php';

/**
 * Aktif kanal ID'sini veritabanından alır
 * @param PDO $pdo Veritabanı bağlantısı
 * @return string|null Aktif kanal ID veya null
 */
function getActiveChannelId($pdo) {
    try {
        $stmt = $pdo->prepare("SELECT setting_value FROM settings WHERE setting_key = 'active_channel_id'");
        $stmt->execute();
        $result = $stmt->fetch();
        return $result['setting_value'] ?? null;
    } catch (PDOException $e) {
        error_log("Aktif kanal ID alma hatası: " . $e->getMessage());
        return null;
    }
}

/**
 * Telegram'a mesaj gönderir
 * @param string $message Mesaj metni
 * @param array $keyboard Butonlar (inline keyboard)
 * @param PDO|null $pdo Veritabanı bağlantısı (aktif kanal ID için)
 * @return array API yanıtı
 */
function sendTelegramMessage($message, $keyboard = null, $pdo = null) {
    // Bot token'ını al
    $botToken = getBotToken($pdo);
    if (empty($botToken)) {
        return [
            'success' => false,
            'error' => 'Bot token bulunamadı. Lütfen admin panelinden ayarlayın.'
        ];
    }
    
    // Aktif kanal ID'sini al
    $chatId = null;
    if ($pdo) {
        $chatId = getActiveChannelId($pdo);
    }
    
    // Eğer veritabanından alınamazsa config'den al
    if (empty($chatId)) {
        $chatId = defined('TELEGRAM_CHAT_ID') && TELEGRAM_CHAT_ID !== 'YOUR_CHAT_ID_HERE' 
            ? TELEGRAM_CHAT_ID 
            : null;
    }
    
    if (empty($chatId)) {
        return [
            'success' => false,
            'error' => 'Aktif kanal ID bulunamadı. Lütfen admin panelinden ayarlayın.'
        ];
    }
    
    $url = "https://api.telegram.org/bot" . $botToken . "/sendMessage";
    
    $data = [
        'chat_id' => $chatId,
        'text' => $message,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => false
    ];
    
    if ($keyboard !== null) {
        $data['reply_markup'] = json_encode([
            'inline_keyboard' => $keyboard
        ]);
    }
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return [
            'success' => false,
            'error' => "HTTP Code: $httpCode",
            'response' => $response
        ];
    }
    
    $result = json_decode($response, true);
    
    return [
        'success' => $result['ok'] ?? false,
        'response' => $result
    ];
}

/**
 * Mesaj.md dosyasını parse edip Telegram formatına çevirir
 * @param string $newDomain Yeni domain (eski domain yerine yazılacak)
 * @return array ['message' => string, 'keyboard' => array]
 */
function parseMessageTemplate($newDomain) {
    $messageFile = __DIR__ . '/mesaj.md';
    
    if (!file_exists($messageFile)) {
        return [
            'message' => "Domain güncellendi: $newDomain",
            'keyboard' => null
        ];
    }
    
    $content = file_get_contents($messageFile);
    $lines = explode("\n", $content);
    
    // '-----------------' satırını bul
    $separatorIndex = -1;
    foreach ($lines as $index => $line) {
        if (strpos(trim($line), '-----------------') !== false) {
            $separatorIndex = $index;
            break;
        }
    }
    
    // Üst kısım (normal mesaj)
    $messageLines = [];
    if ($separatorIndex > 0) {
        $messageLines = array_slice($lines, 0, $separatorIndex);
    } else {
        $messageLines = $lines;
    }
    
    // Mesajı birleştir ve yeni domain'i değiştir
    $message = implode("\n", $messageLines);
    $message = str_replace('https://casinolevant792.com/tr', $newDomain, $message);
    $message = trim($message);
    
    // Alt kısım (butonlar) - separator'dan sonraki satırlar
    $keyboard = null;
    if ($separatorIndex >= 0 && $separatorIndex < count($lines) - 1) {
        $buttonLines = array_slice($lines, $separatorIndex + 1);
        $keyboard = parseButtons($buttonLines, $newDomain);
    }
    
    return [
        'message' => $message,
        'keyboard' => $keyboard
    ];
}

/**
 * Buton satırlarını parse edip Telegram inline keyboard formatına çevirir
 * @param array $buttonLines Buton satırları
 * @param string $newDomain Yeni domain
 * @return array Telegram inline keyboard formatı
 */
function parseButtons($buttonLines, $newDomain) {
    $keyboard = [];
    $rowIndex = 0;
    
    foreach ($buttonLines as $line) {
        $line = trim($line);
        if (empty($line)) continue;
        
        // URL'leri çıkar
        preg_match_all('/\(([^)]+)\)/', $line, $urlMatches);
        $urls = $urlMatches[1] ?? [];
        
        if (empty($urls)) continue;
        
        // İlk iki satır için (Telegram/Canlı Destek ve Instagram/Twitter)
        if ($rowIndex < 2) {
            $row = [];
            
            // Satırı URL'lere göre böl
            $parts = preg_split('/\([^)]+\)/', $line);
            $texts = [];
            foreach ($parts as $part) {
                $text = trim($part);
                $text = preg_replace('/[📱💻🔣]/u', '', $text);
                $text = trim($text);
                if (!empty($text)) {
                    $texts[] = $text;
                }
            }
            
            // Her URL için buton oluştur
            foreach ($urls as $urlIndex => $url) {
                // Metni al
                $text = '';
                if (isset($texts[$urlIndex])) {
                    $text = $texts[$urlIndex];
                } else {
                    // Varsayılan metinler
                    if ($rowIndex === 0) {
                        $text = ($urlIndex === 0) ? 'Telegram' : 'Canlı Destek';
                    } else {
                        $text = ($urlIndex === 0) ? 'Instagram' : 'Twitter';
                    }
                }
                
                // Yeni domain'i değiştir
                $url = str_replace('https://casinolevant792.com/tr', $newDomain, $url);
                
                $row[] = [
                    'text' => $text,
                    'url' => $url
                ];
            }
            
            // İlk iki satır için 2'şer buton ekle
            if (count($row) >= 2) {
                $keyboard[] = array_slice($row, 0, 2);
            } else if (count($row) === 1) {
                $keyboard[] = $row;
            }
            
            $rowIndex++;
        } else {
            // Üçüncü satır - tek uzun buton
            $url = $urls[0] ?? '';
            
            // Metni çıkar
            $parts = preg_split('/\([^)]+\)/', $line);
            $text = '';
            foreach ($parts as $part) {
                $cleanText = trim($part);
                $cleanText = preg_replace('/[📱💻🔣]/u', '', $cleanText);
                $cleanText = trim($cleanText);
                if (!empty($cleanText)) {
                    $text = $cleanText;
                    break;
                }
            }
            
            if (empty($text)) {
                $text = 'Güncel Adresimize Giriş İçin Tıklayın';
            }
            
            // Yeni domain'i değiştir
            $url = str_replace('https://casinolevant792.com/tr', $newDomain, $url);
            
            $keyboard[] = [[
                'text' => $text,
                'url' => $url
            ]];
            
            break; // Üçüncü satırdan sonra dur
        }
    }
    
    return $keyboard;
}

/**
 * Belirli bir kanala Telegram mesajı gönderir
 * @param string $chatId Telegram chat ID
 * @param string $message Mesaj metni
 * @param array|null $keyboard Butonlar
 * @param PDO|null $pdo Veritabanı bağlantısı (bot token için)
 * @return array API yanıtı
 */
function sendTelegramMessageToChannel($chatId, $message, $keyboard = null, $pdo = null) {
    // Bot token'ını veritabanından al
    $botToken = getBotToken($pdo);
    if (empty($botToken)) {
        return [
            'success' => false,
            'error' => 'Telegram bot token ayarlanmamış! Lütfen admin panelinden ayarlayın.'
        ];
    }
    
    $url = "https://api.telegram.org/bot" . $botToken . "/sendMessage";
    
    $data = [
        'chat_id' => $chatId,
        'text' => $message,
        'parse_mode' => 'HTML',
        'disable_web_page_preview' => false
    ];
    
    if ($keyboard !== null) {
        $data['reply_markup'] = json_encode([
            'inline_keyboard' => $keyboard
        ]);
    }
    
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($data),
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 30
    ]);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode !== 200) {
        return [
            'success' => false,
            'error' => "HTTP Code: $httpCode",
            'response' => $response
        ];
    }
    
    $result = json_decode($response, true);
    
    return [
        'success' => $result['ok'] ?? false,
        'response' => $result,
        'error' => $result['ok'] ? null : ($result['description'] ?? 'Bilinmeyen hata')
    ];
}

